/**************************************************************************************
 
   Copyright (c) Hilscher GmbH. All Rights Reserved.
 
 **************************************************************************************
 
   Filename:
    $Id: DeviceHandler.h 12810 2019-03-21 08:55:55Z AlexanderMinor $
   Last Modification:
    $Author: AlexanderMinor $
    $Date: 2019-03-21 09:55:55 +0100 (Do, 21 Mrz 2019) $
    $Revision: 12810 $
   
   Targets:
     Win32/ANSI   : yes
     Win32/Unicode: yes (define _UNICODE)
     WinCE        : no
 
   Description:
     Declaration of the "Device Handler" Class, which handles all physical layers

   Changes:
 
     Version   Date        Author   Description
     ----------------------------------------------------------------------------------
     5         07.10.11    SS       Timeout value for connection establishment added

     4         31.03.11    SS       - Unnecessary functions RemoveAllLayers(), 
                                      LockChannelAccess(), UnlockChannelAccess() removed
     3         02.11.09    PL       Structural change to allow connection handling in 
                                    separate list and devices
     2         03.03.09    MT       Structural change to allow placing physical drivers
                                    in a dll
     
     1         25.09.08    PL       initial version
 
**************************************************************************************/

/////////////////////////////////////////////////////////////////////////////
/// \file DeviceHandler.h
/// Declaration of the "Device Handler" Class, which handles all physical layers
/////////////////////////////////////////////////////////////////////////////

#pragma once

class CTransportLayer;
class CPhysicalInterface;
class CPhysicalLayer;

#include "stdint.h"

#include "APIInterface.h"
#include "ConnectorAPI.h"
#include "CifXErrors.h"
#include "netXAPI.h"

#pragma warning( push )
#pragma warning(disable: 4201)
#include "Mmsystem.h"
#pragma warning( pop )

#include <vector>
#include <deque>
#include <map>

class CEndpoint;
class CDevice;
class CChannel;

/*! Device handler reset command definition */ 
typedef enum DEVICE_HANDLER_RESET_CMD_Etag
{
  eSYSTEMRESET = 1,                   /*!< System reset via system channel          */
  eCHANNELRESET,                      /*!< System reset via communication channel   */
  eCHANNELINIT,                       /*!< Channel init via communication channel   */
  eSYSTEMRESETEX,                     /*!< System reset via system channel and reset mode + parameter */
} DEVICE_HANDLER_RESET_CMD_E;

#define MONITOR_THREAD_TIMEOUT          5000  /* Timeout for monitor thread */
#define MONITOR_DISCONNECT_TIMEOUT      10000 /* Timeout between channel close process and interface connection check */
#define MONITOR_RECONNECT_TIMEOUT       5000  /* Reconnect timeout */
#define CIFX_INIT_TIMEOUT               100   /* Timeout for connection establishment */

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CEndpoint
/// Structure for the handling of a single endpoint
///////////////////////////////////////////////////////////////////////////////////////////
class CEndpoint
{
public:
  CEndpoint(CPhysicalInterface* pcInterface, BYTE bDevice = 0)
    : m_fKeepAliveSupported(false)
    , m_pcInterface(pcInterface)
    , m_pcDefaultDataLayer(NULL)
    , m_ulKeepAliveID(0)
    , m_ulKeepAliveTimeout(HIL_TRANSPORT_KEEP_ALIVE_CLIENT_TIMEOUT)
    , m_bDevice(bDevice)
  {
  }

  ~CEndpoint();

  /* List of all open devices on this endpoint */
  std::map<uint16_t, CDataLayer*> m_cmDataLayers;
  CDataLayer*                   m_pcDefaultDataLayer;
  CPhysicalInterface*           m_pcInterface;

  bool                          m_fKeepAliveSupported;
  uint32_t                      m_ulKeepAliveID;
  uint32_t                      m_ulKeepAliveTimeout;

  BYTE                          m_bDevice;

  CDevice*  GetDeviceObject(uint32_t ulBoard);
  CDevice*  GetDeviceObject(const char* szBoard);
  CDevice*  AddDevice(BOARD_INFORMATION& tBoardInfo);
  uint32_t  GetDeviceCount();

protected:
  std::vector<CDevice*>         m_cvDevices;
};

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CChannel
/// Structure of channel and connection information
///////////////////////////////////////////////////////////////////////////////////////////
class CChannel
{
private:
  CChannel();

public:
  // Constructor
  CChannel(CDevice* pcDevice, uint32_t ulChannel, CIFXHANDLE hChannel, bool fAvailable = true)
    : m_pcDevice(pcDevice)
    , m_lOpenRefCount(0)
    , m_ulChannelNumber(ulChannel)
    , m_hChannel(hChannel)
    , m_fAvailable(fAvailable)
  {
    memset(&m_tChannelInfo, 0, sizeof(m_tChannelInfo));
  }

  // Destructor
  ~CChannel()
  {
  }

  CDevice*                  m_pcDevice;           /*!< Device this channel belongs to */
  int32_t                   m_lOpenRefCount;      /*!< Reference counter of the open connections  */
  uint32_t                  m_ulChannelNumber;    /*!< Channel number                             */
  CIFXHANDLE                m_hChannel;           /*!< Channel handle                             */
  CHANNEL_INFORMATION       m_tChannelInfo;       /*!< Channel information                        */

  // Member functions
  bool                      IsAvailable   ( void)            { return m_fAvailable;       }
  void                      SetAvailable  ( bool fAvailable) { m_fAvailable = fAvailable; }

protected:
  bool                      m_fAvailable;         /*!< Channel available flag         */
};

///////////////////////////////////////////////////////////////////////////////////////////
/// \class CDevice
/// Single instance object a device
///////////////////////////////////////////////////////////////////////////////////////////
class CDevice
{
private:
  CDevice();

public:
  // Constructor
  CDevice(CEndpoint* pcEndpoint, bool fAvailable = true)
    : m_pcEndpoint(pcEndpoint)
    , m_fAvailable(fAvailable)
  {
    memset(&m_tBoardInfo, 0, sizeof(m_tBoardInfo));
  }

  // Destructor
  ~CDevice()
  {
    SetAvailable(false);
    while(m_cmChannels.size() > 0)
    {
      CChannel* pcChannel = m_cmChannels.begin()->second;
      delete pcChannel;

      m_cmChannels.erase(m_cmChannels.begin());
    }
  }

  CEndpoint*                  m_pcEndpoint;         /*!< Endpoint theis boards belongs to           */
  BOARD_INFORMATION           m_tBoardInfo;         /*!< Board information                          */

  CChannel*   AddChannel        ( uint32_t ulChannel, CHANNEL_INFORMATION* ptChannelInfo);
  CChannel*   GetChannelObject  ( uint32_t ulChannel);
  int32_t     EnumChannel       ( uint32_t ulChannel, CHANNEL_INFORMATION* ptChannelInfo);
  bool        IsAvailable       ( void)             { return m_fAvailable;       }
  
  bool  IsAccessed()
  {
    bool fRet = false;
    std::map<uint32_t, CChannel*>::iterator iter = m_cmChannels.begin();

    while(iter != m_cmChannels.end())
    {
      if (0 != iter->second->m_lOpenRefCount)
      {
        fRet = true;
        break;
      }
      iter++;
    }
    
    return fRet;
  }

  void  SetAvailable(bool fAvailable) 
  { 
    m_fAvailable = fAvailable;

    if(!fAvailable)
    {
      /* Device has gone, so invalidate channels */
      std::map<uint32_t, CChannel*>::iterator iter = m_cmChannels.begin();

      while(iter != m_cmChannels.end())
      {
        iter->second->SetAvailable(fAvailable);
        iter++;
      }
    }
  }

protected:
  std::map<uint32_t, CChannel*>  m_cmChannels;       /*!< Channel map for this device */
  bool                           m_fAvailable;       /*!< Channel available flag       */
};


///////////////////////////////////////////////////////////////////////////////////////////
/// \class CDeviceHandler
/// Single instance object which handles all attached physical drivers, and
/// creates need transport layers on device attach, etc.
///////////////////////////////////////////////////////////////////////////////////////////
class CDeviceHandler
{
protected:
  typedef enum DEVICE_STATE_Etag
  {
    eONLINE,
    ePAUSED,
    eOFFLINE,
  } DEVICE_STATE_E;

  /////////////////////////////////////////////////////
  ///  Definition of the interface data structure 
  /////////////////////////////////////////////////////
  typedef struct INTERFACE_DATA_Ttag
  {
    CPhysicalInterface*     pcInterface;
    std::vector<CEndpoint*> cvEndpoints;
    DEVICE_STATE_E          eState;
  } INTERFACE_DATA_T;

public:
  CDeviceHandler(void);
  virtual ~CDeviceHandler(void);

  /////////////////////////////////////////////////////
  ///  Definition of the notification data structure 
  /////////////////////////////////////////////////////
  typedef struct NOTIFICATION_DATA_Ttag
  {
    INTERFACE_DATA_T*             ptInterface;                      /*!< Name of the inteface                  */
    NETX_INTERFACE_NOTIFICATION_E eNotify;                          /*!< Notification state                    */
  } NOTIFICATION_DATA_T, *PNOTIFICATION_DATA_T;
  
  typedef enum INTERFACE_MONITOR_COMMAND_Etag
  {
    eDISCONNECT,                         /*!< Schedule disconnect interface   */
    eRECONNECT,                          /*!< Schedule reconnect interface    */
    eRESET,                              /*!< Reset interface monitor         */
  } INTERFACE_MONITOR_COMMAND_E;

  typedef struct INTERFACE_MONITOR_DATA_Ttag
  {
    uint32_t                    ulTimer;
    INTERFACE_MONITOR_COMMAND_E eCmd;

  } INTERFACE_MONITOR_DATA_T;

  typedef std::map<std::string, PNOTIFICATION_DATA_T> NOTIFICATION_MAP;
  
  /* Helper functions for cifX API functions */
  int32_t            EnumerateDevice       ( uint32_t ulBoard, BOARD_INFORMATION* ptBoardInfo);
  int32_t            EnumerateChannels     ( uint32_t ulBoard, uint32_t ulChannel, CHANNEL_INFORMATION* ptChannelInfo);
  int32_t            BrowseDevices         ( std::vector<BOARD_INFORMATION>& cvBoardList, PFN_NXAPI_BROWSE_CALLBACK pfnCallback = NULL, void* pvUser = NULL );

  uint32_t        GetDeviceCount        ( void);
  CDevice*        GetDevice             ( uint32_t ulBoard, bool fOpenDevice = false);
  CDevice*        GetDevice             ( char* szName);
  CPhysicalLayer* GetLayer              ( const char* szLayerPrefix);
  CPhysicalLayer* GetLayer              ( uint32_t ulIdx);

  static void     CreateInterfaceName   ( std::string&   szInterfaceName,
                                          char*          pszEnpointName,
                                          uint32_t  ulNameBufferSize,
                                          char*          pszNameBuffer);
  static bool     SplitDeviceName(char* szFullName, std::string& szInterface, std::string& szBoard);

  int32_t         OpenDeviceConnection  ( char* szBoard, uint32_t ulChannel, CIFXHANDLE* phChannel);

  int32_t         CloseDeviceConnection ( HANDLE hDevCon);

  int32_t         Init                  ( void);
  void            Deinit                ( void);

  int32_t         HandleReset           ( DEVICE_HANDLER_RESET_CMD_E  eResetCmd,
                                          CChannel*                   pcChannel,
                                          uint32_t                    ulTimeout,
                                          uint32_t                    ulMode);

  ///////////////////////////////////////
  // Physical Layer specific functions //
  ///////////////////////////////////////

  int32_t        AddLayer                   ( CPhysicalLayer* pcLayer);

  void           HandleInterfaceNotification( std::string& szInterface, NETX_INTERFACE_NOTIFICATION_E eNotify, CPhysicalLayer* pcLayer);

  void           ScheduleInterfaceMonitor   ( std::string& szInterface, INTERFACE_MONITOR_COMMAND_E eCmd);

  ///////////////////////////////////////////////////////////////////////////////////////////
  /// Search an endpoint
  ///   \param szInterface    Interface name
  ///   \param bDevice        Device number of the interface
  ///   \return CIFX_NO_ERROR on success
  ///////////////////////////////////////////////////////////////////////////////////////////
  CEndpoint* GetEndpoint(std::string szInterface, BYTE bDevice)
  {
    CEndpoint*  pcRet = NULL;

    std::map<std::string, INTERFACE_DATA_T>::iterator iter = m_cmInterfaces.find(szInterface);

    if(iter != m_cmInterfaces.end())
    {
      INTERFACE_DATA_T& tInterface = iter->second;

      if(bDevice < tInterface.cvEndpoints.size())
      {
        pcRet = tInterface.cvEndpoints[bDevice];
      }
    }
    return pcRet;
  }

protected:  
  // Map of available interfaces
  typedef std::map<std::string, INTERFACE_DATA_T>         INTERFACE_MAP;
  typedef std::map<std::string, INTERFACE_MONITOR_DATA_T> INTERFACE_MONITOR_QUEUE;

  INTERFACE_MAP                     m_cmInterfaces;                 /*!< Map of connected interfaces  */
  CRITICAL_SECTION                  m_tcsInterfaceLock;             /*!< Interface locking            */

  std::vector<CPhysicalLayer*>      m_cvLayers;                     /*!< Array of all registered layers */

  /* Interface Notification */
  std::deque<NOTIFICATION_DATA_T>   m_cvInterfaceNotifications;
  CRITICAL_SECTION                  m_tcsNotificationLock;
  HANDLE                            m_hInterfaceNotifyThread;
  HANDLE                            m_hInterfaceNotifySemaphore;
  bool                              m_fInterfaceNotifyStop;

  static DWORD WINAPI     InterfaceNotifyThread      (void* pvParam);
  static DWORD WINAPI     InterfaceMonitorThread     (void* pvParam);
  void                    InterfaceMonitorThreadFunc (void);

  HANDLE                  m_hInterfaceMonitorThread;
  HANDLE                  m_hInterfaceMonitorStop;
  INTERFACE_MONITOR_QUEUE m_cmInterfaceMonitorQueue;
  CRITICAL_SECTION        m_tcsInterfaceMonitorQueue;

  // Member fnctions
  int32_t               EstablishConnection     (INTERFACE_DATA_T& tInterface);    
  void                  ReleaseConnection       (INTERFACE_DATA_T& tInterface);
  void                  InterfaceDisconnect     (INTERFACE_DATA_T& tInterface);
};
